package com.orhon.smartcampus.material.gql.scalars.datetime;

import graphql.Internal;
import graphql.language.StringValue;
import graphql.schema.*;

import java.time.DateTimeException;
import java.time.OffsetTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.TemporalAccessor;
import java.util.function.Function;

import static com.orhon.smartcampus.material.gql.scalars.util.Kit.typeName;


/**
 * Access this via {@link static com.orhon.smartcampus.modules.gql.scalars.ExtendedScalars#Time}
 */
@Internal
public class TimeScalar extends GraphQLScalarType {

    private final static DateTimeFormatter dateFormatter = DateTimeFormatter.ISO_OFFSET_TIME;

    public TimeScalar() {
        super("Time", "An RFC-3339 compliant Full Time Scalar", new Coercing<OffsetTime, String>() {
            @Override
            public String serialize(Object input) throws CoercingSerializeException {
                TemporalAccessor temporalAccessor;
                if (input instanceof TemporalAccessor) {
                    temporalAccessor = (TemporalAccessor) input;
                } else if (input instanceof String) {
                    temporalAccessor = parseOffsetTime(input.toString(), CoercingSerializeException::new);
                } else {
                    throw new CoercingSerializeException(
                            "Expected a 'String' or 'java.time.temporal.TemporalAccessor' but was '" + typeName(input) + "'."
                    );
                }
                try {
                    return dateFormatter.format(temporalAccessor);
                } catch (DateTimeException e) {
                    throw new CoercingSerializeException(
                            "Unable to turn TemporalAccessor into full time because of : '" + e.getMessage() + "'."
                    );
                }
            }

            @Override
            public OffsetTime parseValue(Object input) throws CoercingParseValueException {
                TemporalAccessor temporalAccessor;
                if (input instanceof TemporalAccessor) {
                    temporalAccessor = (TemporalAccessor) input;
                } else if (input instanceof String) {
                    temporalAccessor = parseOffsetTime(input.toString(), CoercingParseValueException::new);
                } else {
                    throw new CoercingParseValueException(
                            "Expected a 'String' or 'java.time.temporal.TemporalAccessor' but was '" + typeName(input) + "'."
                    );
                }
                try {
                    return OffsetTime.from(temporalAccessor);
                } catch (DateTimeException e) {
                    throw new CoercingParseValueException(
                            "Unable to turn TemporalAccessor into full time because of : '" + e.getMessage() + "'."
                    );
                }
            }

            @Override
            public OffsetTime parseLiteral(Object input) throws CoercingParseLiteralException {
                if (!(input instanceof StringValue)) {
                    throw new CoercingParseLiteralException(
                            "Expected AST type 'StringValue' but was '" + typeName(input) + "'."
                    );
                }
                return parseOffsetTime(((StringValue) input).getValue(), CoercingParseLiteralException::new);
            }

            private OffsetTime parseOffsetTime(String s, Function<String, RuntimeException> exceptionMaker) {
                try {
                    TemporalAccessor temporalAccessor = dateFormatter.parse(s);
                    return OffsetTime.from(temporalAccessor);
                } catch (DateTimeParseException e) {
                    throw exceptionMaker.apply("Invalid RFC3339 full time value : '" + s + "'. because of : '" + e.getMessage() + "'");
                }
            }
        });
    }

}
