
def xmodule_cmd(watch=false, debug=false)
    xmodule_cmd = 'xmodule_assets common/static/xmodule'
    if watch
        "watchmedo shell-command " +
                   "--patterns='*.js;*.coffee;*.sass;*.scss;*.css' " +
                   "--recursive " +
                   "--command='#{xmodule_cmd}' " +
                   "common/lib/xmodule"
    else
        xmodule_cmd
    end
end

def coffee_cmd(watch=false, debug=false)
    if watch
        # On OSx, coffee fails with EMFILE when
        # trying to watch all of our coffee files at the same
        # time.
        #
        # Ref: https://github.com/joyent/node/issues/2479
        #
        # Instead, watch 50 files per process in parallel
        cmds = []
        Dir['*/static/**/*.coffee'].each_slice(50) do |coffee_files|
            cmds << "node_modules/.bin/coffee --watch --compile #{coffee_files.join(' ')}"
        end
        cmds
    else
        'node_modules/.bin/coffee --compile */static'
    end
end

def sass_cmd(watch=false, debug=false)
    "sass #{debug ? '--debug-info' : '--style compressed'} " +
          "--load-path ./common/static/sass " +
          "--require ./common/static/sass/bourbon/lib/bourbon.rb " +
          "#{watch ? '--watch' : '--update'} */static"
end

desc "Compile all assets"
multitask :assets => 'assets:all'

namespace :assets do

    desc "Compile all assets in debug mode"
    multitask :debug

    desc "Watch all assets for changes and automatically recompile"
    task :watch => 'assets:_watch' do
        puts "Press ENTER to terminate".red
        $stdin.gets
    end

    {:xmodule => :install_python_prereqs,
     :coffee => :install_node_prereqs,
     :sass => :install_ruby_prereqs}.each_pair do |asset_type, prereq_task|
        desc "Compile all #{asset_type} assets"
        task asset_type => prereq_task do
            cmd = send(asset_type.to_s + "_cmd", watch=false, debug=false)
            if cmd.kind_of?(Array)
                cmd.each {|c| sh(c)}
            else
                sh(cmd)
            end
        end

        multitask :all => asset_type
        multitask :debug => "assets:#{asset_type}:debug"
        multitask :_watch => "assets:#{asset_type}:_watch"

        namespace asset_type do
            desc "Compile all #{asset_type} assets in debug mode"
            task :debug => prereq_task do
                cmd = send(asset_type.to_s + "_cmd", watch=false, debug=true)
                sh(cmd)
            end

            desc "Watch all #{asset_type} assets and compile on change"
            task :watch => "assets:#{asset_type}:_watch" do
                puts "Press ENTER to terminate".red
                $stdin.gets
            end

            task :_watch => prereq_task do
                cmd = send(asset_type.to_s + "_cmd", watch=true, debug=true)
                if cmd.kind_of?(Array)
                    cmd.each {|c| background_process(c)}
                else
                    background_process(cmd)
                end
            end
        end
    end


    multitask :sass => 'assets:xmodule'
    namespace :sass do
        # In watch mode, sass doesn't immediately compile out of date files,
        # so force a recompile first
        task :_watch => 'assets:sass:debug'
        multitask :debug => 'assets:xmodule:debug'
    end

    multitask :coffee => 'assets:xmodule'
    namespace :coffee do
        multitask :debug => 'assets:xmodule:debug'
    end
end

[:lms, :cms].each do |system|
    # Per environment tasks
    environments(system).each do |env|
        desc "Compile coffeescript and sass, and then run collectstatic in the specified environment"
        task "#{system}:gather_assets:#{env}" => :assets do
            sh("#{django_admin(system, env, 'collectstatic', '--noinput')} > /dev/null") do |ok, status|
                if !ok
                    abort "collectstatic failed!"
                end
            end
        end
    end
end
