"""
Internationalization tasks
"""
import sys
import subprocess
from path import path
from paver.easy import task, cmdopts, needs, sh
from pavelib.utils.envs import Env

try:
    from pygments.console import colorize
except ImportError:
    colorize = lambda color, text: text  # pylint: disable-msg=invalid-name


@task
@needs(
    "pavelib.i18n.i18n_validate_gettext",
    "pavelib.assets.compile_coffeescript",
)
@cmdopts([
    ("verbose", "v", "Sets 'verbose' to True"),
])
def i18n_extract(options):
    """
    Extract localizable strings from sources
    """
    verbose = getattr(options, "verbose", None)
    cmd = Env.REPO_ROOT / "i18n" / "extract.py"

    if verbose:
        cmd += " -vv"

    sh(cmd)


@task
@needs("pavelib.i18n.i18n_extract")
def i18n_generate():
    """
    Compile localizable strings from sources, extracting strings first.
    """
    cmd = Env.REPO_ROOT / "i18n" / "generate.py"
    sh(cmd)


@task
@needs("pavelib.i18n.i18n_extract")
def i18n_generate_strict():
    """
    Compile localizable strings from sources, extracting strings first.
    Complains if files are missing.
    """
    cmd = Env.REPO_ROOT / "i18n" / "generate.py"
    sh(cmd + " --strict")


@task
@needs("pavelib.i18n.i18n_extract")
def i18n_dummy():
    """
    Simulate international translation by generating dummy strings
    corresponding to source strings.
    """
    cmd = Env.REPO_ROOT / "i18n" / "dummy.py"
    sh(cmd)


@task
def i18n_validate_gettext():
    """
    Make sure GNU gettext utilities are available
    """

    returncode = subprocess.call(['which', 'xgettext'])

    if returncode != 0:
        msg = colorize(
            'red',
            "Cannot locate GNU gettext utilities, which are "
            "required by django for internationalization.\n (see "
            "https://docs.djangoproject.com/en/dev/topics/i18n/"
            "translation/#message-files)\nTry downloading them from "
            "http://www.gnu.org/software/gettext/ \n"
        )

        sys.stderr.write(msg)
        sys.exit(1)


@task
def i18n_validate_transifex_config():
    """
    Make sure config file with username/password exists
    """
    home = path('~').expanduser()
    config = home / '.transifexrc'

    if not config.isfile or config.getsize == 0:
        msg = colorize(
            'red',
            "Cannot connect to Transifex, config file is missing"
            " or empty: {config} \nSee "
            "http://help.transifex.com/features/client/#transifexrc \n".format(
                config=config,
            )
        )

        sys.stderr.write(msg)
        sys.exit(1)


@task
@needs("pavelib.i18n.i18n_validate_transifex_config")
def i18n_transifex_push():
    """
    Push source strings to Transifex for translation
    """
    cmd = Env.REPO_ROOT / "i18n" / "transifex.py"
    sh("{cmd} push".format(cmd=cmd))


@task
@needs("pavelib.i18n.i18n_validate_transifex_config")
def i18n_transifex_pull():
    """
    Pull translated strings from Transifex
    """
    cmd = Env.REPO_ROOT / "i18n" / "transifex.py"
    sh("{cmd} pull".format(cmd=cmd))


@task
@needs(
    "pavelib.i18n.i18n_transifex_pull",
    "pavelib.i18n.i18n_extract",
    "pavelib.i18n.i18n_dummy",
    "pavelib.i18n.i18n_generate_strict",
)
def i18n_robot_pull():
    """
    Pull source strings, generate po and mo files, and validate
    """
    sh('git clean -fdX conf/locale')
    sh('paver test_i18n')
    sh('git add conf/locale')

    sh(
        'git commit --message="Update translations '
        '(autogenerated message)" --edit'
    )


@task
@needs(
    "pavelib.i18n.i18n_extract",
    "pavelib.i18n.i18n_transifex_push",
)
def i18n_robot_push():
    """
    Extract new strings, and push to transifex
    """
    pass
