"""
Schema for validating and sanitizing data received from the JavaScript client.
"""

import dateutil
from pytz import utc
from voluptuous import Schema, Required, All, Any, Range, In, Invalid
from openassessment.xblock.xml import parse_examples_xml_str, UpdateFromXmlError


def utf8_validator(value):
    """Validate and sanitize unicode strings.
    If we're given a bytestring, assume that the encoding is UTF-8

    Args:
        value: The value to validate

    Returns:
        unicode

    Raises:
        Invalid

    """
    try:
        if isinstance(value, str):
            return value.decode('utf-8')
        else:
            return unicode(value)
    except (ValueError, TypeError):
        raise Invalid(u"Could not load unicode from value \"{val}\"".format(val=value))


def datetime_validator(value):
    """Validate and sanitize a datetime string in ISO format.

    Args:
        value: The value to validate

    Returns:
        unicode: ISO-formatted datetime string

    Raises:
        Invalid

    """
    try:
        # The dateutil parser defaults empty values to the current day,
        # which is NOT what we want.
        if value is None or value == '':
            raise Invalid(u"Datetime value cannot be \"{val}\"".format(val=value))

        # Parse the date and interpret it as UTC
        value = dateutil.parser.parse(value).replace(tzinfo=utc)
        return unicode(value.isoformat())
    except (ValueError, TypeError):
        raise Invalid(u"Could not parse datetime from value \"{val}\"".format(val=value))


def examples_xml_validator(value):
    """Parse and validate student training examples XML.

    Args:
        value: The value to parse.

    Returns:
        list of training examples, serialized as dictionaries.

    Raises:
        Invalid

    """
    try:
        return parse_examples_xml_str(value)
    except UpdateFromXmlError:
        raise Invalid(u"Could not parse examples from XML")


# Schema definition for an update from the Studio JavaScript editor.
EDITOR_UPDATE_SCHEMA = Schema({
    Required('prompt'): utf8_validator,
    Required('title'): utf8_validator,
    Required('feedback_prompt'): utf8_validator,
    Required('submission_start'): Any(datetime_validator, None),
    Required('submission_due'): Any(datetime_validator, None),
    Required('assessments'): [
        Schema({
            Required('name'): All(
                utf8_validator,
                In([
                    u'peer-assessment',
                    u'self-assessment',
                    u'example-based-assessment',
                    u'student-training'
                ])
            ),
            Required('start', default=None): Any(datetime_validator, None),
            Required('due', default=None): Any(datetime_validator, None),
            'must_grade': All(int, Range(min=0)),
            'must_be_graded_by': All(int, Range(min=0)),
            'examples': All(utf8_validator, examples_xml_validator)
        })
    ],
    Required('feedbackprompt', default=u""): utf8_validator,
    Required('criteria'): [
        Schema({
            Required('order_num'): All(int, Range(min=0)),
            Required('name'): utf8_validator,
            Required('prompt'): utf8_validator,
            Required('feedback'): All(
                utf8_validator,
                In([
                    'disabled',
                    'optional',
                    'required',
                ])
            ),
            Required('options'): [
                Schema({
                    Required('order_num'): All(int, Range(min=0)),
                    Required('name'): utf8_validator,
                    Required('explanation'): utf8_validator,
                    Required('points'): All(int, Range(min=0)),
                })
            ]
        })
    ]
})