"""The edx_lint write command."""
from __future__ import print_function

import os

from six.moves import configparser
from six.moves import cStringIO
from io import TextIOWrapper

import pkg_resources

from edx_lint.tamper_evident import TamperEvidentFile
from edx_lint.configfile import merge_configs


WARNING_HEADER = """\
# ***************************
# ** DO NOT EDIT THIS FILE **
# ***************************
#
# It is generated by:
#   $ edx_lint write pylintrc
#
#
#
#
#
#
#
#
# STAY AWAY!
#
#
#
#
#
# SERIOUSLY.
#
# ------------------------------
"""

def write_main(argv):
    if len(argv) != 1:
        print("Please provide the name of a file to write.")
        return 1

    filename = argv[0]
    resource_name = "files/" + filename
    tweaks_name = amend_filename(filename, "_tweaks")

    if not pkg_resources.resource_exists("edx_lint", resource_name):
        print("Don't have file %r to write." % filename)
        return 2

    if os.path.exists(filename):
        print("Checking existing copy of %s" % filename)
        tef = TamperEvidentFile(filename)
        if not tef.validate():
            bak_name = amend_filename(filename, "_backup")
            print("Your copy of %s seems to have been edited, renaming it to %s" % (filename, bak_name))
            if os.path.exists(bak_name):
                print("A previous %s exists, deleting it" % bak_name)
                os.remove(bak_name)
            os.rename(filename, bak_name)

    print("Reading edx_lint/files/%s" % filename)
    cfg = configparser.RawConfigParser()

    # pkg_resources always reads binary data (in both python2 and python3).
    # ConfigParser.read_string only exists in python3, so we have to wrap the string
    # from pkg_resources in a cStringIO so that we can pass it into ConfigParser.readfp.
    resource_string = pkg_resources.resource_string("edx_lint", resource_name).decode("ascii")
    cfg.readfp(cStringIO(resource_string), resource_name)

    if os.path.exists(tweaks_name):
        print("Applying local tweaks from %s" % tweaks_name)
        cfg_tweaks = configparser.RawConfigParser()
        cfg_tweaks.read([tweaks_name])

        merge_configs(cfg, cfg_tweaks)

    print("Writing %s" % filename)
    output_text = cStringIO()
    output_text.write(WARNING_HEADER)
    cfg.write(output_text)

    out_tef = TamperEvidentFile(filename)
    out_tef.write(output_text.getvalue())

    return 0


def amend_filename(filename, amend):
    """Amend a filename with a suffix.

    amend_filename("foo.txt", "_tweak") --> "foo_tweak.txt"

    """
    base, ext = os.path.splitext(filename)
    amended_name = base + amend + ext
    return amended_name
