"""
End to end test of event exports.
"""

import gzip
import os
import logging
import tempfile
import textwrap
import time
import shutil

from luigi.s3 import S3Client, S3Target

from edx.analytics.tasks.tests.acceptance import AcceptanceTestCase
from edx.analytics.tasks.url import url_path_join


log = logging.getLogger(__name__)


class EventExportAcceptanceTest(AcceptanceTestCase):
    """Validate data flow for bulk export of events for research purposes."""

    INPUT_FILE = 'event_export_tracking.log'
    PROD_SERVER_NAME = 'prod-edxapp-001'
    EDGE_SERVER_NAME = 'prod-edge-edxapp-002'
    NUM_REDUCERS = 1

    def setUp(self):
        super(EventExportAcceptanceTest, self).setUp()

        # The name of an existing job flow to run the test on
        assert('job_flow_name' in self.config)
        # The git URL of the repository to checkout analytics-tasks from.
        assert('tasks_repo' in self.config)
        # The branch of the analytics-tasks repository to test. Note this can differ from the branch that is currently
        # checked out and running this code.
        assert('tasks_branch' in self.config)
        # Where to store logs generated by analytics-tasks.
        assert('tasks_log_path' in self.config)
        # The user to connect to the job flow over SSH with.
        assert('connection_user' in self.config)
        # Where analytics-tasks should output data, should be a URL pointing to a directory.
        assert('tasks_output_url' in self.config)
        # Allow for parallel execution of the test by specifying a different identifier. Using an identical identifier
        # allows for old virtualenvs to be reused etc, which is why a random one is not simply generated with each run.
        assert('identifier' in self.config)

        url = self.config['tasks_output_url']
        identifier = self.config['identifier']

        self.s3_client = S3Client()

        self.test_root = url_path_join(url, identifier, 'event_export')

        self.s3_client.remove(self.test_root, recursive=True)

        self.test_src = url_path_join(self.test_root, 'src')
        self.test_out = url_path_join(self.test_root, 'out')
        self.test_config_root = url_path_join(self.test_root, 'config')

        self.test_config = url_path_join(self.test_config_root, 'default.yaml')
        self.test_gpg_key_dir = url_path_join(self.test_config_root, 'gpg-keys')

        self.input_paths = {
            'prod': url_path_join(self.test_src, self.PROD_SERVER_NAME, 'tracking.log-20140515.gz'),
            'edge': url_path_join(self.test_src, self.EDGE_SERVER_NAME, 'tracking.log-20140516-12345456.gz')
        }

        self.upload_data()
        self.write_config()
        self.upload_public_keys()

    def upload_data(self):
        src = os.path.join(self.data_dir, 'input', self.INPUT_FILE)

        with tempfile.NamedTemporaryFile(suffix='.gz') as temp_file:
            gzip_file = gzip.open(temp_file.name, 'wb')
            try:
                with open(src, 'r') as input_file:
                    shutil.copyfileobj(input_file, gzip_file)
            finally:
                gzip_file.close()

            temp_file.flush()

            # Upload test data file
            for dst in self.input_paths.values():
                self.s3_client.put(temp_file.name, dst)

    def write_config(self):
        with S3Target(self.test_config).open('w') as target_file:
            target_file.write(
                textwrap.dedent(
                    """
                    ---
                    environments:
                      prod:
                        servers:
                          - {server_1}
                      edge:
                        servers:
                          - {server_2}
                    organizations:
                      edX:
                        recipient: daemon@edx.org
                      AcceptanceX:
                        recipient: daemon+2@edx.org
                    """
                    .format(
                        server_1=self.PROD_SERVER_NAME,
                        server_2=self.EDGE_SERVER_NAME
                    )
                )
            )

    def upload_public_keys(self):
        gpg_key_dir = os.path.join('gpg-keys')
        for key_filename in os.listdir(gpg_key_dir):
            full_local_path = os.path.join(gpg_key_dir, key_filename)
            remote_url = url_path_join(self.test_gpg_key_dir, key_filename)

            if not key_filename.endswith('.key'):
                self.s3_client.put(full_local_path, remote_url)

    def test_event_log_exports_using_manifest(self):
        with tempfile.NamedTemporaryFile() as temp_config_file:
            temp_config_file.write(
                textwrap.dedent(
                    """
                    [manifest]
                    threshold = 1
                    """
                )
            )
            temp_config_file.flush()

            self.launch_task(config=temp_config_file.name)
            self.validate_output()

    def launch_task(self, config=None):
        command = [
            os.getenv('REMOTE_TASK'),
            '--job-flow-name', self.config.get('job_flow_name'),
            '--branch', self.config.get('tasks_branch'),
            '--repo', self.config.get('tasks_repo'),
            '--remote-name', self.config.get('identifier'),
            '--wait',
            '--log-path', self.config.get('tasks_log_path'),
            '--user', self.config.get('connection_user'),
        ]

        if config:
            command.extend(['--override-config', config])

        command.extend(
            [
                'EventExportTask',
                '--local-scheduler',
                '--source', self.test_src,
                '--output-root', self.test_out,
                '--delete-output-root',
                '--config', self.test_config,
                '--environment', 'prod',
                '--environment', 'edge',
                '--interval', '2014-05',
                '--gpg-key-dir', self.test_gpg_key_dir,
                '--gpg-master-key', 'daemon+master@edx.org',
                '--n-reduce-tasks', str(self.NUM_REDUCERS),
            ]
        )

        self.call_subprocess(command)

    def validate_output(self):
        for server_id in [self.PROD_SERVER_NAME, self.EDGE_SERVER_NAME]:
            for use_master_key in [False, True]:
                self.validate_output_file('2014-05-15', 'edX', server_id, use_master_key)
                self.validate_output_file('2014-05-16', 'edX', server_id, use_master_key)
                self.validate_output_file('2014-05-15', 'AcceptanceX', server_id, use_master_key)

    def validate_output_file(self, day, org_id, server_id, use_master_key=False):
        if use_master_key:
            key_filename = 'insecure_master_secret.key'
        else:
            if org_id == 'edX':
                key_filename = 'insecure_secret.key'
            else:
                key_filename = 'insecure_secret_2.key'

        self.temporary_dir = tempfile.mkdtemp()
        self.addCleanup(shutil.rmtree, self.temporary_dir)

        self.downloaded_outputs = os.path.join(self.temporary_dir, 'output')
        os.makedirs(self.downloaded_outputs)

        local_file_name = '{day}_{org_id}.log'.format(day=day, org_id=org_id)
        remote_url = url_path_join(self.test_out, org_id, server_id, local_file_name + '.gpg')

        # Files won't appear in S3 instantaneously, wait for the files to appear.
        # TODO: exponential backoff
        for _i in range(30):
            key = self.s3_client.get_key(remote_url)
            if key is not None:
                break
            else:
                time.sleep(2)

        if key is None:
            self.fail('Unable to find expected output file {0}'.format(remote_url))

        downloaded_output_path = os.path.join(self.downloaded_outputs, remote_url.split('/')[-1])
        key.get_contents_to_filename(downloaded_output_path)

        decrypted_file_name = downloaded_output_path[:-len('.gpg')]
        self.decrypt_file(downloaded_output_path, decrypted_file_name, key_filename)

        self.call_subprocess(['diff', decrypted_file_name, os.path.join(self.data_dir, 'output', local_file_name)])
