"""
Create archive files containing log files for a particular organization.
"""
import datetime
import logging
import os
import shutil
import tempfile
import tarfile

import luigi
import yaml

from edx.analytics.tasks.event_exports import EventExportTask
from edx.analytics.tasks.mapreduce import MultiOutputMapReduceJobTask
from edx.analytics.tasks.url import get_target_from_url, url_path_join, ExternalURL, IgnoredTarget
from edx.analytics.tasks.util.tempdir import make_temp_directory
from edx.analytics.tasks.pathutil import PathSetTask

log = logging.getLogger(__name__)


class ArchiveExportTask(MultiOutputMapReduceJobTask):
    """
    Groups source URLs by destination organization and outputs to a single tar file per org.

    Parameters:
        eventlog_output_root: Directory to find the output of intermediate event-logs-by-org in.
        config: url path to configuration file that defines organizations and their aliases.
        output_root: url path to location where output archives get written.
        temp_dir: optional path to local file directory to use to create archives.
        org_id: A list of organizations to process data for. If provided, only these organizations will be processed.
            Otherwise, all valid organizations will be processed.

    """
    eventlog_output_root = luigi.Parameter(
        default_from_config={'section': 'event-export', 'name': 'output_root'}
    )
    config = luigi.Parameter(
        default_from_config={'section': 'event-export', 'name': 'config'}
    )
    output_root = luigi.Parameter(
        default_from_config={'section': 'archive-event-export', 'name': 'output_root'}
    )
    temp_dir = luigi.Parameter(default=None)
    org_id = luigi.Parameter(is_list=True, default=[])

    # Force this job to flush each counter increment instead of
    # batching them. The tasks does not output data directly through
    # Hadoop. Since there is no other way for Hadoop to verify that
    # the task is progressing, we use counters as a keep-alive.
    batch_counter_default = 1

    # For the reason described above we copy to S3 in chunks, using a counter between chunks.
    # This is the size of each chunk
    chunk_size = 16 * 1024 * 1024  # 16 MB

    def requires(self):
        return ExternalURL(self.config)

    def output(self):
        # Because this task writes to a shared directory, we don't
        # want to include a marker for job success.  Use a special
        # target that always triggers new runs and never writes out.
        return IgnoredTarget()

    def input_hadoop(self):
        # The contents of the config will be passed to the mapper method,
        # so that it gets called but without a lot of wasted I/O.
        # But we do not want the source files (as generated by
        # an upstream workflow) to be passed to the mapper, so isolate
        # this from requires().
        return get_target_from_url(self.config)

    def jobconfs(self):
        # Make sure that only one mapper runs.  The mapper generates all
        # reducer input in a single call, so it should be called once
        # from a single mapper node.  Appending should override any previous
        # settings of the parameter (as per java command line).
        jcs = super(ArchiveExportTask, self).jobconfs()
        jcs.append('mapred.map.tasks=1')
        return jcs

    def init_local(self):
        self._get_organization_info()
        self.date_label = "{date}".format(date=datetime.date.today())

    def init_mapper(self):
        self.mapper_done = False

    def _generate_log_files_for_org(self, org_name):
        """
        A generator that yields log files for the specified org.

        Yields tuple of absolute and relative paths (relative to org subdirectory in source).
        """

        # If org_ids are specified, restrict the processed files to that set.
        if len(self.org_id) > 0 and org_name not in self.org_id:
            return

        org_source = url_path_join(self.eventlog_output_root, org_name)

        # Only include paths that include ".log" so that directory names are not included.
        task = PathSetTask(src=org_source, include=['*.log*'])
        for target in task.output():
            target_url = target.path
            relative_url = target_url[len(self.eventlog_output_root):].lstrip('/')
            yield (target_url, relative_url)

    def mapper(self, _line):
        # Enforce that this code gets run once and only once.
        if self.mapper_done:
            return

        for org_primary_name in self.organizations:
            for target_url, relative_url in self._generate_log_files_for_org(org_primary_name):
                yield org_primary_name, (target_url, relative_url)
            for alt_name in self.organizations[org_primary_name].get('other_names', []):
                for target_url, relative_url in self._generate_log_files_for_org(alt_name):
                    yield org_primary_name, (target_url, relative_url)

        self.mapper_done = True

    def extra_modules(self):
        import gnupg
        return [gnupg, yaml]

    def output_path_for_key(self, key):
        primary_org_id = key
        tar_name = '{0}-{1}-tracking.tar'.format(self.date_label, primary_org_id)
        return url_path_join(self.output_root, tar_name)

    def multi_output_reducer(self, key, values, output_file):
        log.debug("Calling reducer for key %s.", key)
        primary_org_id = key

        # Make sure that parent temp directory exists.  We're going to leave it around
        # when done, because it may be used for other things.
        if not os.path.isdir(self.temp_dir):
            log.info("Local temporary directory does not exist, creating: %s", self.temp_dir)
            os.makedirs(self.temp_dir)

        # Create a temp file to contain the archive output locally.
        with make_temp_directory(prefix="archive", dir=self.temp_dir) as tar_temp_dir:
            temp_tar_filepath = os.path.join(tar_temp_dir, "{org}.tar".format(org=primary_org_id))

            # Python 2.6 doesn't have support for context manager form, so do it ourselves.
            tar_output_file = tarfile.open(temp_tar_filepath, mode='w')
            try:
                self._write_files_to_tarfile(tar_output_file, values, tar_temp_dir)
            finally:
                tar_output_file.close()

            # When done writing the file locally, move it to its final destination.
            with open(temp_tar_filepath, 'r') as src_file:
                self._copy_file(src_file, output_file)

    def _get_organization_info(self):
        """Pulls organization configuration out of .yaml file."""
        with self.input_hadoop().open() as config_input:
            config_data = yaml.load(config_input)
        self.organizations = config_data['organizations']

    def _write_files_to_tarfile(self, tar_output_file, urls, tar_temp_dir):
        """Writes files to open tarfile, using temp directory for local storage."""
        for target_url, relative_url in urls:
            log.debug("Processing %s", target_url)

            source_target = get_target_from_url(target_url)
            # Copy file to local temp file.
            with tempfile.NamedTemporaryFile(dir=tar_temp_dir, delete=False) as temp_input_file:
                local_temp_filepath = temp_input_file.name
                with source_target.open('r') as source_file:
                    self._copy_file(source_file, temp_input_file)

            # Once we have added the temp file to the tarfile, we don't need it anymore.
            tar_output_file.add(name=local_temp_filepath, arcname=relative_url)
            os.remove(local_temp_filepath)

            # See comment for `batch_counter_default` above.
            self.incr_counter('Archive exported events', 'Files archived', 1)

    def _copy_file(self, source, destination):
        """Copy a file object to another in predetermined chunk size"""
        while True:
            data = source.read(self.chunk_size)
            if data:
                # See comment for `batch_counter_default` above.
                self.incr_counter('Archive exported events', 'Bytes transferred', len(data))
                destination.write(data)
            else:
                break


class ArchivedEventExportWorkflow(ArchiveExportTask):
    """
    Generates archive (tar) files containing log files per organization.

    A few parameters are shared by :py:class:`EventExportTask` and :py:class:`ArchiveExportTask`:

        eventlog_output_root: Directory to write intermediate event-logs-by-org to.
        config: A URL to a YAML file that contains the list of organizations and servers to export events for.

    Additional parameters are passed through to :py:class:`ArchiveExportTask`:

        output_root: url path to location where output archives get written.
        temp_dir: optional path to local file directory to use to create archives.

    Several parameters are passed through to :py:class:`EventExportTask`.  These are:

        source: A URL to a path that contains log files that contain the events.
        environment: A list of short strings that describe the environment that generated the events. Only include
            events from this list of environments.
        interval: The range of dates to export logs for.
        pattern: A regex with a named capture group for the date that approximates the date that the events within were
            emitted. Note that the search interval is expanded, so events don't have to be in exactly the right file
            in order for them to be processed.

    Additional optional parameters are passed through to :py:class:`MapReduceJobTask`:

        mapreduce_engine:  'hadoop' (the default) or 'local'.
        n_reduce_tasks: number of reducer tasks to use in upstream tasks.

    Additional optional parameters are passed through to :py:class:`MultiOutputMapReduceJobTask`:

        delete_output_root: if True, recursively deletes the output_root at task creation.

    """

    # Define parameters that are used only for EventExportTask.
    source = luigi.Parameter(
        default_from_config={'section': 'event-logs', 'name': 'source'}
    )
    environment = luigi.Parameter(is_list=True, default=['prod', 'edge'])
    interval = luigi.DateIntervalParameter()
    pattern = luigi.Parameter(default=None)

    def requires(self):
        return EventExportTask(
            output_root=self.eventlog_output_root,
            config=self.config,
            source=self.source,
            environment=self.environment,
            interval=self.interval,
            pattern=self.pattern,
            org_id=self.org_id,
            mapreduce_engine=self.mapreduce_engine,
            n_reduce_tasks=self.n_reduce_tasks,
            delete_output_root=self.delete_output_root,
        )
