define([
        'jquery',
        'moment',
        'underscore',
        'collections/product_collection',
        'models/course_model',
        'models/course_seats/professional_seat',
        'jquery-cookie'
    ],
    function ($,
              moment,
              _,
              ProductCollection,
              Course,
              ProfessionalSeat) {
        'use strict';

        var model,
            honorSeat = {
                id: 8,
                url: 'http://ecommerce.local:8002/api/v2/products/8/',
                structure: 'child',
                product_class: 'Seat',
                title: 'Seat in edX Demonstration Course with honor certificate',
                price: '0.00',
                expires: null,
                attribute_values: [
                    {
                        name: 'certificate_type',
                        value: 'honor'
                    },
                    {
                        name: 'course_key',
                        value: 'edX/DemoX/Demo_Course'
                    },
                    {
                        name: 'id_verification_required',
                        value: false
                    }
                ],
                is_available_to_buy: true
            },
            verifiedSeat = {
                id: 9,
                url: 'http://ecommerce.local:8002/api/v2/products/9/',
                structure: 'child',
                product_class: 'Seat',
                title: 'Seat in edX Demonstration Course with verified certificate (and ID verification)',
                price: '15.00',
                expires: null,
                attribute_values: [
                    {
                        name: 'certificate_type',
                        value: 'verified'
                    },
                    {
                        name: 'course_key',
                        value: 'edX/DemoX/Demo_Course'
                    },
                    {
                        name: 'id_verification_required',
                        value: true
                    }
                ],
                is_available_to_buy: true
            },
            creditSeat = {
                id: 10,
                url: 'http://ecommerce.local:8002/api/v2/products/10/',
                structure: 'child',
                product_class: 'Seat',
                title: 'Seat in edX Demonstration Course with credit certificate (and ID verification)',
                price: '200.00',
                expires: null,
                attribute_values: [
                    {
                        name: 'certificate_type',
                        value: 'credit'
                    },
                    {
                        name: 'course_key',
                        value: 'edX/DemoX/Demo_Course'
                    },
                    {
                        name: 'id_verification_required',
                        value: true
                    },
                    {
                        name: 'credit_provider',
                        value: 'Harvard'
                    },
                    {
                        name: 'credit_hours',
                        value: 1
                    }
                ],
                is_available_to_buy: true
            },
            alternateCreditSeat = {
                id: 11,
                url: 'http://ecommerce.local:8002/api/v2/products/11/',
                structure: 'child',
                product_class: 'Seat',
                title: 'Seat in edX Demonstration Course with credit certificate (and ID verification)',
                price: '300.00',
                expires: null,
                attribute_values: [
                    {
                        name: 'certificate_type',
                        value: 'credit'
                    },
                    {
                        name: 'course_key',
                        value: 'edX/DemoX/Demo_Course'
                    },
                    {
                        name: 'id_verification_required',
                        value: true
                    },
                    {
                        name: 'credit_provider',
                        value: 'MIT'
                    },
                    {
                        name: 'credit_hours',
                        value: 2
                    }
                ],
                is_available_to_buy: true
            },
            data = {
                id: 'edX/DemoX/Demo_Course',
                url: 'http://ecommerce.local:8002/api/v2/courses/edX/DemoX/Demo_Course/',
                name: 'edX Demonstration Course',
                verification_deadline: '2015-10-01T00:00:00Z',
                type: 'credit',
                products_url: 'http://ecommerce.local:8002/api/v2/courses/edX/DemoX/Demo_Course/products/',
                last_edited: '2015-07-27T00:27:23Z',
                products: [
                    honorSeat,
                    verifiedSeat,
                    creditSeat,
                    alternateCreditSeat,
                    {
                        id: 7,
                        url: 'http://ecommerce.local:8002/api/v2/products/7/',
                        structure: 'parent',
                        product_class: 'Seat',
                        title: 'Seat in edX Demonstration Course',
                        price: null,
                        expires: null,
                        attribute_values: [
                            {
                                name: 'course_key',
                                value: 'edX/DemoX/Demo_Course'
                            }
                        ],
                        is_available_to_buy: false
                    }
                ]
            };

        describe('Course model', function () {
            beforeEach(function () {
                model = Course.findOrCreate(data, {parse: true});

                // Remove the parent products
                model.removeParentProducts();
            });

            describe('removeParentProducts', function () {
                it('should remove all parent products from the products collection', function () {
                    var products;

                    // Re-initialize the model since the beforeEach removes the parents
                    model = Course.findOrCreate(data, {parse: true});

                    // Sanity check to ensure the products were properly parsed
                    products = model.get('products');
                    expect(products.length).toEqual(5);

                    // Remove the parent products
                    model.removeParentProducts();

                    // Only the children survived...
                    expect(products.length).toEqual(4);
                    expect(products.where({structure: 'child'}).length).toEqual(4);
                });
            });

            // NOTE (CCB): There is a bug preventing this from being called 'toJSON'.
            // See https://github.com/karma-runner/karma/issues/1534.
            describe('#toJSON', function () {
                it('should not modify verification_deadline if verification_deadline is empty', function () {
                    var json,
                        values = [null, ''];

                    _.each(values, function (value) {
                        model.set('verification_deadline', value);
                        json = model.toJSON();
                        expect(json.verification_deadline).toEqual(value);
                    });
                });

                it('should add a timezone to verification_deadline if verification_deadline is not empty', function () {
                    var json,
                        deadline = '2015-01-01T00:00:00';

                    model.set('verification_deadline', deadline);
                    json = model.toJSON();

                    expect(json.verification_deadline).toEqual(deadline + '+00:00');
                });
            });

            describe('save', function () {
                var expectedAjaxData = function (data) {
                    var products,
                        expected = {
                            id: data.id,
                            name: data.name,
                            verification_deadline: moment.utc(data.verification_deadline).format()
                        };

                    products = _.filter(data.products, function (product) {
                        return product.structure === 'child';
                    });

                    expected.products = _.map(products, function (product) {
                        return product;
                    });

                    return expected;
                };

                it('should POST to the publication endpoint', function () {
                    var args,
                        cookie = 'save-test';

                    spyOn($, 'ajax');
                    $.cookie('ecommerce_csrftoken', cookie);

                    expect(model.validate()).toBeFalsy();
                    model.save();

                    // $.ajax should have been called
                    expect($.ajax).toHaveBeenCalled();

                    // Ensure the data was POSTed to the correct endpoint
                    args = $.ajax.calls.argsFor(0)[0];
                    expect(args.type).toEqual('POST');
                    expect(args.url).toEqual('/api/v2/publication/');
                    expect(args.contentType).toEqual('application/json');
                    expect(args.headers).toEqual({'X-CSRFToken': cookie});
                    expect(JSON.parse(args.data)).toEqual(expectedAjaxData(data));
                });
            });

            describe('getOrCreateSeats', function () {
                it('should return existing seats', function () {
                    var mapping = {
                            'honor': [honorSeat],
                            'verified': [verifiedSeat],
                            'credit': [creditSeat, alternateCreditSeat]
                        },
                        seats;

                    _.each(mapping, function (expected, seatType) {
                        seats = model.getOrCreateSeats(seatType);

                        _.each(seats, function (seat) {
                            expect(expected).toContain(seat.toJSON());
                        });
                    });
                });

                it('should return an empty array if an audit seat does not already exist', function () {
                    expect(model.getOrCreateSeats('audit')).toEqual([]);
                });

                it('should create a new CourseSeat if one does not exist', function () {
                    var seat;

                    // Sanity check to confirm a new seat is created later
                    expect(model.seats().length).toEqual(4);

                    // A new seat should be created
                    seat = model.getOrCreateSeats('professional')[0];
                    expect(model.seats().length).toEqual(5);

                    // The new seat's class/type should correspond to the passed in seat type
                    expect(seat).toEqual(jasmine.any(ProfessionalSeat));
                });
            });

            describe('products', function () {
                it('is a ProductCollection', function () {
                    expect(model.get('products')).toEqual(jasmine.any(ProductCollection));
                });
            });

            describe('verification deadline validation', function () {
                it('succeeds if the verification deadline is after the course seats\' expiration dates', function () {
                    var seat = model.getOrCreateSeats('verified')[0];
                    model.set('verification_deadline', '2016-01-01T00:00:00Z');
                    seat.set('expires', '2015-01-01T00:00:00Z');

                    expect(model.validate()).toBeUndefined();
                    expect(model.isValid(true)).toBeTruthy();
                });

                it('fails if the verification deadline is before the course seats\' expiration dates', function () {
                    var seat = model.getOrCreateSeats('verified')[0],
                        msg = 'The verification deadline must occur AFTER the upgrade deadline.';
                    model.set('verification_deadline', '2014-01-01T00:00:00Z');
                    seat.set('expires', '2015-01-01T00:00:00Z');

                    expect(model.validate().verification_deadline).toEqual(msg);
                    expect(model.isValid(true)).toBeFalsy();
                });
            });

            describe('products validation', function () {
                describe('with single value', function () {
                    it('should return an error message if any product is invalid', function () {
                        var msg = 'Product validation failed.',
                            products = model.get('products');

                        // Add an invalid product
                        products.push(new ProfessionalSeat({price: null}));

                        expect(model.validate().products).toEqual(msg);
                        expect(model.isValid(true)).toBeFalsy();
                    });
                });

                describe('with non-products', function () {
                    it('should have an undefined return value', function () {
                        expect(model.validation.products([])).toBeUndefined();
                    });
                });
            });
        });
    }
);
